package org.msh.tb.bd.tbforms.indicator.tb12;

import org.jboss.seam.annotations.In;
import org.msh.tb.bd.entities.enums.SmearStatus;
import org.msh.tb.bd.tbforms.indicator.tb11.TBForm11;
import org.msh.tb.bd.tbforms.query.TbFormQuery;
import org.msh.tb.bd.tbforms.query.tb12.TB12Column;
import org.msh.tb.bd.tbforms.query.tb12.TB12Row;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.Tbunit;
import org.msh.tb.entities.enums.CaseState;
import org.msh.tb.entities.enums.Gender;
import org.msh.tb.entities.enums.PatientType;
import org.msh.tb.indicators.core.Indicator2D;

import javax.persistence.EntityManager;
import java.util.List;
import java.util.Map;

/**
 * @author MSANTOS
 * Parent abstract class that contain all logic needed to generates the 3 tables for TB 11 Form
 */
 public abstract class TBForm12 extends Indicator2D{

    @In(create=true) Map<String, String> messages;

    /**
     * Method that insert the values on the tables based on the types defined to rows and TB12Columns.
     * This method also calculates the total row and the grand-total TB12Column.
     * @param c TB12Column that will receive the quantity
     * @param row TB12Row that will receive the quantity
     * @param gender Gender sub TB12Column that will receive the quantity
     * @param qtd the quantity that will be set
     */
    protected void setValue(TB12Column c, TB12Row row, Gender gender, Float qtd){
        if(gender == null || row == null || c == null || qtd == null)
            return;

        String g = (gender.equals(Gender.MALE) ? messages.get("manag.gender.male") : messages.get("manag.gender.female"));

        addValue(c.name() + g, g, messages.get(row.getKey()), qtd);
        addValue(c.name() + g, g, messages.get(TB12Row.TOTAL.getKey()), qtd);

        if(c.equals(TB12Column.TOTAL_REGISTERED)){
            addValue(TB12Column.TOTAL_REGISTERED.name()+"T", messages.get("manag.pulmonary.sum"), messages.get(row.getKey()), qtd);
            addValue(TB12Column.TOTAL_REGISTERED.name()+"T", messages.get("manag.pulmonary.sum"), messages.get(TB12Row.TOTAL.getKey()), qtd);
        }else{
            addValue(TB12Column.GRAND_TOTAL.name() + g, g, messages.get(row.getKey()), qtd);
            addValue(TB12Column.GRAND_TOTAL.name() + g, g, messages.get(TB12Row.TOTAL.getKey()), qtd);
            addValue(TB12Column.GRAND_TOTAL.name() + "T", messages.get("manag.pulmonary.sum"), messages.get(row.getKey()), qtd);
            addValue(TB12Column.GRAND_TOTAL.name() + "T", messages.get("manag.pulmonary.sum"), messages.get(TB12Row.TOTAL.getKey()), qtd);
        }
    }

    /**
     * Method that creates all cells on the report with the start value as 0
     */
    protected void createInterfaceFields(){
        for(TB12Row r : TB12Row.values()){
            for(TB12Column c : TB12Column.values()){
                addValue(c.name()+"M", messages.get("manag.gender.male"), messages.get(r.getKey()), new Float(0));
                addValue(c.name()+"F", messages.get("manag.gender.female"), messages.get(r.getKey()), new Float(0));

                if(c.equals(TB12Column.TOTAL_REGISTERED) || c.equals(TB12Column.GRAND_TOTAL))
                    addValue(c.name()+"T", messages.get("manag.pulmonary.sum"), messages.get(r.getKey()), new Float(0));
            }
        }
    }

    /**
     * Execute the needed queries to generate the table
     */
    protected void createIndicators() {
        TbFormQuery formQuery = getFormQuery();

        createInterfaceFields();

        Tbunit tbunit = getIndicatorFilters().getTbunitselection().getSelected();
        AdministrativeUnit adminUnit = getIndicatorFilters().getTbunitselection().getLastLevelAdminUnitSelected();

        List<Object[]> result = formQuery.queryDB(tbunit,
                adminUnit,
                getWorkspace(),
                getIndicatorFilters().getQuarter(),
                getEntityManager());

        for (Object[] o : result) {
            int queryType = (Integer) o[0];

            switch(queryType) {
                case 0:
                    loadSmearValue(o);
                    break;
                case 1:
                    loadOutcomeValue(o);
                    break;
                case 2:
                    loadTotalRegisteredValue(o);
                    break;
            }
        }
    }

    private void loadSmearValue(Object[] o) {
        PatientType pt = (PatientType) o[1];
        PatientType prevPt = (PatientType) o[2];
        Gender g = (Gender) o[3];
        SmearStatus smear = (SmearStatus) o[4];
        Long qtd = (Long) o[5];
        TB12Column c = TB12Column.getSmearTB12Column(smear);

        if(c == null) {
            throw new RuntimeException("TB12Column must not be null.");
        }

        setValue(c, TB12Row.getPatientTypeAsRow(pt.equals(PatientType.PREVIOUSLY_TREATED) ? prevPt : pt), g, qtd.floatValue());
    }

    private void loadOutcomeValue(Object[] o) {
        PatientType pt = (PatientType) o[1];
        PatientType prevPt = (PatientType) o[2];
        Gender g = (Gender) o[3];
        CaseState caseState = (CaseState) o[4];
        Long qtd = (Long) o[5];
        TB12Column c = TB12Column.getOutcomeTB12Column(caseState);

        if(c == null) {
            throw new RuntimeException("TB12Column must not be null.");
        }

        setValue(c, TB12Row.getPatientTypeAsRow(pt.equals(PatientType.PREVIOUSLY_TREATED) ? prevPt : pt), g, qtd.floatValue());
    }

    private void loadTotalRegisteredValue(Object[] o) {
        PatientType pt = (PatientType) o[1];
        PatientType prevPt = (PatientType) o[2];
        Gender g = (Gender) o[3];
        Long qtd = (Long) o[4];

        setValue(TB12Column.TOTAL_REGISTERED, TB12Row.getPatientTypeAsRow(pt.equals(PatientType.PREVIOUSLY_TREATED) ? prevPt : pt), g, qtd.floatValue());
    }

    protected abstract TbFormQuery getFormQuery();
}
